<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Classes;
use App\Models\ExamAnswerSheet;
use App\Models\Lesson;
use App\Models\ExamControl;
use App\Models\ExamSubject;
use App\Models\ExamType;
use App\Models\Staff;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use PhpOffice\PhpSpreadsheet\Calculation\Category;

class ExamController extends Controller
{
    protected $layout = 'admin::pages.examControl.';
     public function __construct()
    {
        $this->middleware('permission:exam-control-view', ['only' => ['index']]);
        $this->middleware('permission:exam-control-create', ['only' => ['onSave']]);
        $this->middleware('permission:exam-control-destroy', ['only' => ['onDelete']]);
        $this->middleware('permission:exam-control-view-info', ['only' => ['onView']]);
    }
    
    public function index(Request $req)
    {
        $data['data'] = ExamControl::whereClassId($req->class_id)
            ->orderByDesc("id")
            ->get();
        $data['exam_types'] = ExamType::where('status', 1)->get();
        $data['staffs'] = Staff::all();
        $data['class'] = Classes::find($req->class_id);
        $lesson_id = [];
        if ($data['class']->class_structure_program) {
            $lesson_id = $data['class']->class_structure_program->structure_program->lesson_id;
        }
        $data['lessons'] = Lesson::whereIn('id', $lesson_id)->get();
        return view($this->layout . 'index', $data);
    }
    
    public function onSave(Request $req)
    {
        $id = $req->exam_id;
        $item = [
            'class_id' => $req->class_id,
            'exam_type_id' => $req->exam_type_id,
            'exam_date' => (new Carbon($req->exam_date))->format('Y-m-d H:i:s'),
            'duration' => $req->exam_duration,
            'noted' => $req->exam_noted,
        ];
        $req->validate([
            "exam_type_id" => 'required',
            "exam_date" => 'required',
            "exam_duration"  => "required",
        ]);
        DB::beginTransaction();
        try {
            if (!$id) {
                ExamControl::create($item);
                $status = 'Created success';
            } else {
                ExamControl::find($id)->update($item);
                $status = 'Updated success';
            }
            DB::commit();
            return response()->json([
                'code' => 20000,
                'message' => $status,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'code' => 40005,
                'message' => 'Created unsuccess!',
            ]);
        }
    }
    public function onDelete()
    {
        $data = ExamControl::findOrFail(request('id'));
        DB::beginTransaction();
        try {
            $data->delete();
            if ($data->exam_subjects && count($data->exam_subjects) > 0) {
                foreach ($data->exam_subjects as $exam_subject) {
                    $exam_subject->delete();
                    if ($exam_subject->exam_answer_sheets && count($exam_subject->exam_answer_sheets) > 0) {
                        foreach ($exam_subject->exam_answer_sheets as $eas) {
                            $eas->delete();
                        }
                    }
                }
            }
            DB::commit();
            Session::flash('success', 'Delete success.');
            return redirect()->back();
        } catch (\Exception $e) {
            DB::rollBack();
            Session::flash('warning', 'Deleted unsuccess!');
            return redirect()->back();
        }
    }
    public function onView()
    {
        $data = ExamControl::with('exam_type')
            ->with('exam_subjects', function ($q) {
                $q->with('staff');
                $q->with('exam_answer_sheets', function ($qq) {
                    $qq->with('student_lesson_score', function ($qqq) {
                        $qqq->with('lesson');
                    });
                });
            })
            ->with('exam_candidates', function ($q) {
                $q->with('student');
            })
            ->findOrFail(request('exam_id'));
        $data['exam_date_only'] = (new Carbon($data->exam_date))->format('Y-m-d');
        return response()->json($data);
    }
    public function onEnterStudentScore(Request $req)
    {
        $data['data'] = Classes::findOrFail($req->class_id);
        $data['lessons'] = Lesson::whereIn('id', $data['data']->class_structure_program->structure_program->lesson_id)->get();
        return view($this->layout . 'studentScore.enter-student-score', $data);
    }
}
