<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Classes;
use App\Models\ClassOccupation;
use App\Models\Lesson;
use App\Models\LessonOccupation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Session;
use Exception;
use App\Models\StructureProgram;
use Nette\Schema\Elements\Structure;
use App\Models\ClassStructureProgram;

class StructureProgramController extends Controller
{
    protected $layout = 'admin::pages.structureProgram.';
    public function __construct()
    {
        parent::__construct();
        $this->middleware('permission:structure-program-view', ['only' => ['index']]);
        $this->middleware('permission:structure-program-create', ['only' => ['onCreate', 'onSave']]);
        $this->middleware('permission:structure-program-update', ['only' => ['onCreate', 'onSave']]);
        $this->middleware('permission:structure-program-assign', ['only' => ['onAssignClass', 'onSaveAssignClas']]);
        $this->middleware('permission:structure-program-delete', ['only' => ['onDelete']]);
        $this->middleware('permission:structure-program-unassign', ['only' => ['onRemoveClass']]);
    }
    public function index(Request $req)
    {
        if (!$req->id) {
            return redirect()->route('admin-structure-program-list', 1);
        }
        $data['lessons'] = Lesson::all();
        $data['lesson_occupations'] = LessonOccupation::where('status', 1)->get();
        $data['class_occupations'] = ClassOccupation::where('status', 1)->get();
        $data['data'] = StructureProgram::when(filled(request('search')), function ($q) {
            $q->where(function ($q) {
                $q->where('description', 'like', '%' . request('search') . '%');
            });
        })
            ->when($this->userBranch, function ($q) {
                $q->whereHas('user', function ($q) {
                    $q->where('branch_id', $this->userBranch->id);
                });
            })
            ->orderByDesc("id")
            ->paginate(50);
        return view($this->layout . 'index', $data);
    }
    public function onCreate(Request $req)
    {
        $data["data"] = StructureProgram::find($req->id);
        $data['lessons'] = Lesson::where('status', 1)->get();
        $data['lesson_occupations'] = LessonOccupation::where('status', 1)->get();
        $data['class_occupations'] = ClassOccupation::where('status', 1)->get();
        if (isset($data['data']) && $data['data']->class_structure_programs) {
            return redirect()->route("admin-structure-program-list", 1);
        } else {
            if (isset($data['data'])) {
                $data['data']['lessons'] = lesson::with('occupation')->whereIn('id', $data['data']->lesson_id)->get();
            }
            return view($this->layout . 'create', $data);
        }
    }

    public function onSave(Request $req)
    {
        try {
            $id = $req->id;
            $req->validate([
                "class_occupation_id" => "required",
                "lesson_id" => "required",
            ], [
                "class_occupation_id.required" => "Please input this field",
                "lesson_id.required" => "Please input this field",
            ]);
            $item = [
                "class_occupation_id" => $req->class_occupation_id,
                "lesson_id" => array_map('intval', explode(',', $req->lesson_id)),
                "description" => $req->description,
                "user_id" => auth()->user()->id,
            ];
            $status = "Create success.";
            DB::beginTransaction();
            if (!$id) {
                StructureProgram::create($item);
            } else {
                StructureProgram::find($id)->update($item);
                $status = "Update success.";
            }
            DB::commit();
            Session::flash("success", $status);
            return redirect()->route("admin-structure-program-list", 1);
        } catch (Exception $error) {
            DB::rollback();
            Session::flash('warning', 'Create unsuccess!');
            return redirect()->back();
        }
    }

    public function onAssignClass(Request $req)
    {

        $data['data'] = StructureProgram::find($req->id);
        $data['classes'] = Classes::whereIn('class_status_id', [1, 2])->orderByDesc("id")->get();
        if ($data['data']->class_structure_programs) {
            return redirect()->route("admin-structure-program-list", 1);
        } else {
            return view($this->layout . 'assign-class', $data);
        }
    }

    public function onSaveAssignClass(Request $req)
    {
        $item = [
            'class_id' => $req->class_id,
            'structure_program_id' => $req->id,
        ];
        DB::beginTransaction();
        try {
            ClassStructureProgram::create($item);
            DB::commit();
            Session::flash("success", 'Assign class success');
            return redirect()->route("admin-structure-program-list", 1);
        } catch (Exception $error) {
            DB::rollback();
            Session::flash('warning', 'Assign class unsuccess!');
            return redirect()->back();
        }
    }
    public function onRemoveClass(Request $req)
    {
        $data = StructureProgram::find($req->id);
        DB::beginTransaction();
        try {
            $csp = ClassStructureProgram::where('structure_program_id', $data->id)->where('class_id', $data->class_structure_program->class_id)->first();
            if ($req->status) {
                $classes = Classes::findOrFail($data->class_structure_program->class_id);
                $classes->students()->detach();
            }
            $csp->delete();
            DB::commit();
            return response()->json([
                'code' => 20000,
                'message' => 'Remove assigned class success',
            ]);
        } catch (Exception $error) {
            DB::rollback();
            return response()->json([
                'code' => 40005,
                'message' => 'Remove assigned class unsuccess!',
            ]);
        }
    }

    public function onDelete(Request $req)
    {
        $data = StructureProgram::find($req->id);
        DB::beginTransaction();
        try {
            $data->delete();
            DB::commit();
            Session::flash("success", 'Remove structure program success');
            return redirect()->route("admin-struture-program-list", 1);
        } catch (Exception $error) {
            DB::rollback();
            Session::flash('warning', 'Remove structure program unsuccess!');
            return redirect()->back();
        }
    }

    public function lesson()
    {
        $data = Lesson::query()
            ->with('occupation')
            ->when(filled(request('search')), function ($q) {
                $q->where(function ($q) {
                    $q->where('name', 'like', '%' . request('search') . '%')
                        ->orWhere('short_name', 'like', '%' . request('search') . '%');
                });
            })
            ->whereHas('occupation', function ($q) {
                $q->whereStatus(1);
            })
            ->whereStatus(1)
            ->orderByDesc('id')
            ->paginate(25);

        return response()->json($data);
    }
}