<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Attendance;
use App\Models\ClassStructureProgram;
use App\Models\Classes;
use App\Models\ExamAnswerSheet;
use App\Models\ExamCandidate;
use App\Models\ExamControl;
use App\Models\ExamSubject;
use App\Models\Group;
use App\Models\Lesson;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;


class TeacherPortalController extends Controller
{
    protected $layout = "admin::pages.teacherPortal.";
    
    public function checkedUser(int $class_id)
    {
        $user = Classes::find($class_id)->users()->get();
        $allow_user_id = [];
        foreach ($user as $item) {
            $allow_user_id[] = $item->id;
        }
        return $allow_user_id;
    }
    
    // main page in teacher portal
    public function index()
    {
        $data['data'] = ClassStructureProgram::when(filled(request('keyword')), function ($q) {
            $q->where(function ($q) {
                $q->where('class_id', 'like', '%' . request('keyword') . '%');
            });
        })
            ->select('id', 'class_id', 'structure_program_id')
            ->orderByDesc("class_id")
            ->get();
        $data['data_id'] = ClassStructureProgram::select('class_id')->get();
        if (request('first20') && request('first20') == 'true') {
            $data['data'] = ClassStructureProgram::all()->take(20);
        }
        return view($this->layout . 'table', $data);
    }

    // view
    public function oninsertAttendance(Request $req)
    {
        $data['user_checked'] = $this->checkedUser($req->class_id);
        if ($req->id) {
            $data['attendance'] = Attendance::findOrFail($req->id);
        }
        $classes = Classes::find($req->class_id);
        $csp = ClassStructureProgram::where('class_id', $req->class_id)->first();
        $data['lessons'] = Lesson::all();
        $data['students'] = [];
        $data['lessons_id'] = [];
        if (isset($csp->structure_program)) {
            $data['lessons_id'] = $csp->structure_program->lesson_id;
        }
        if (isset($classes->students)) {
            $data['students'] = $classes->students;
        }
        return view($this->layout . 'attendance', $data);
    }

    public function onSaveAttendance(Request $req)
    {
        if (in_array(auth()->user()->id, $this->checkedUser($req->class_id))) {
            $att = Attendance::find($req->id);
            $item = [
                "class_id" => $req->class_id,
                "lesson_id" => $req->lesson_id,
                "student_id" => $req->student_id ? array_map('intval', $req->student_id) : null,
                "is_attended" => 1,
                "log_date" => $req->id ? $att->log_date : Carbon::now()->format('Y-m-d H:i:s'),
            ];
            $status = "Create success.";
            DB::beginTransaction();
            try {
                if ($req->id) {
                    $status = 'Update success';
                    $att->update($item);
                } else {
                    Attendance::create($item);
                }
                DB::commit();
                Session::flash("success", $status);
                if ($req->id) {
                    return redirect()->route("admin-teacher-portal-attend", $req->class_id);
                } else {
                    return redirect()->route("admin-teacher-portal-list");
                }
            } catch (\Exception $error) {
                DB::rollback();
                Session::flash('warning', 'Create unsuccess!');
                return redirect()->back();
            }
        } else {
            Session::flash('warning', 'Unautherized user');
            return redirect()->route("admin-teacher-portal-list");
        }
    }

    // today attend
    public function todayAttend(Request $req)
    {
        $data['today'] = Carbon::now()->format('Y-m-d');
        $data['data'] = Attendance::where('class_id', $req->class_id)->whereDate('log_date', $data['today'])->orderByDesc("id")->get();
        $data['students'] = Classes::find($req->class_id)->students()->get();
        return view($this->layout . 'attend', $data);
    }

    // student list in each class

    public function studentList(Request $req)
    {
        $data['data'] = Classes::find($req->class_id)->students()->get();
        return view($this->layout . 'student', $data);
    }


    // class attendance
    public function classAttendance(Request $req)
    {
        $data['attendance'] = Attendance::where('class_id', $req->class_id)->get();
        $data['classes'] = Classes::find($req->class_id);
        $data['lesson_id'] = $data['classes']->class_structure_program->structure_program->lesson_id;
        $data['data'] = Lesson::whereIn('id', $data['lesson_id'])->with('attendance')->get();
        return view($this->layout . 'class-attendance', $data);
    }

    // group list 
    public function groupList(Request $req)
    {
        $data['data'] = Group::where('class_id', $req->class_id)->get();
        $data['count'] = count(Classes::find($req->class_id)->students()->get());
        return view($this->layout . 'group', $data);
    }

    // additional information
    public function additionalInfo(Request $req)
    {
        $data['data'] = Classes::find($req->class_id);
        return view($this->layout . 'additional-info', $data);
    }

    // exam control 
    public function exam(Request $req)
    {
        $data['user_checked'] = $this->checkedUser($req->class_id);
        $data['data'] = ExamControl::where('class_id', $req->class_id)->get();
        return view($this->layout . 'exam', $data);
    }
    // exam subject
    public function examSubject()
    {
        $data = ExamSubject::where('exam_id', request('exam_id'))->get();
        return response()->json($data);
    }
    // exam candidate
    public function examCandidate()
    {
        $data = ExamCandidate::where('exam_id', request('exam_id'))->with('student')->get();
        return response()->json($data);
    }
    // exam answer sheet
    public function examAnswerSheet()
    {
        if (request('class_id')) {
            $data = ExamAnswerSheet::where('class_id', request('class_id'))->with('exam_candidate', function ($q) {
                $q->with('student');
            })
                ->get();
            return response()->json($data);
        }
        $data = ExamAnswerSheet::where('exam_subject_id', request('exam_subject_id'))->with('exam_candidate', function ($q) {
            $q->with('student');
        })
            ->get();;
        return response()->json($data);
    }
    // student score
    public function studentScore()
    {
        $data = ExamAnswerSheet::where('id', request('id'))->with('exam_subject', function ($q) {
            $q->with('staff');
        })
            ->with('exam_candidate', function ($q) {
                $q->with('student');
            })
            ->first();
        return response()->json($data);
        // return $data;
    }

    // orally question
    public function onExamAnswerSheet(Request $req)
    {
        if (in_array(auth()->user()->id, $this->checkedUser($req->class_id))) {
            $score = 0;
            if ($req->score > 100) {
                $score = 100;
            } elseif ($req->score < 0) {
                $score = 0;
            } else {
                $score = $req->score;
            }
    
            $item = [
                'class_id' => $req->class_id,
                'exam_subject_id' => $req->exam_subject,
                'exam_candidate_id' => $req->student,
                'acquired_score' => $score,
                'answer_sheet' => $req->answer_sheet,
            ];
            $req->validate([
                'answer_sheet' => 'nullable',
                'score' => 'min:0|max:100',
            ]);
            DB::beginTransaction();
            try {
                ExamAnswerSheet::updateOrCreate(
                    [
                        'class_id' => $req->class_id,
                        'exam_subject_id' => $req->exam_subject,
                        'exam_candidate_id' => $req->student,
                    ],
                    $item
                );
                DB::commit();
                return response()->json([
                    'status' => true,
                    'message' => 'Create success',
                ]);
            } catch (\Exception $e) {
                return $e;
                DB::rollBack();
                return response()->json([
                    'status' => false,
                    'message' => 'Create unsuccess!'
                ]);
            }
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Unautherized user'
            ]);
        }
    }
}
