<?php

namespace App\Http\Controllers\website;

use App\Http\Controllers\Controller;
use App\Http\Controllers\WebsiteBaseController;
use App\Http\Requests\Website\ResetPasswordRequest;
use App\Http\Requests\Website\SignInRequest;
use App\Http\Requests\Website\SignUpRequest;
use App\Models\EducationLevel;
use App\Models\Student;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class AuthController extends WebsiteBaseController
{
    protected $layout = 'website::pages.auth.';

    public function onSignIn()
    {
        return view($this->layout. 'sign-in');
    }

    public function onAccessSignIn(SignInRequest $request)
    {
        try {
            $remember   = $request->remember ? true : false;
            $status     = 'success';
            $error      = false;
            $message    = 'Login success';
            $redirectTo = request('returnUrl') ?? route('website-home');
            $user       = Student::whereEmail($request->email)
                            ->first();

            switch ($user) {
                case $user->role != 'student';
                    $status     = 'info';
                    $error      = true;
                    $message    = 'Your account is not a student account';
                    break;
                case $user->status != config('dummy.status.active'):
                    $status     = 'info';
                    $error      = true;
                    $message    = 'Your account is not active';
                    break;

                case Hash::check($request->password, $user->password) == false:
                    $status     = 'error';
                    $error      = true;
                    $message    = 'Password is not correct';
                    break;

                default:
                    auth('web')->login($user, $remember);
                    break;
            }

            return response()->json([
                'status'   => $status,
                'error'    => $error,
                'message'  => $message,
                'redirectTo' => $redirectTo,
            ]);

        } catch (\Exception $e) { 
            return response()->json([
                'status'   => 'error',
                'message'  => 'Your account is not found',
                'error'    => true,
            ]);
        }
    }

    public function onSignUp()
    {
        return view($this->layout.'sign-up');
    }

    public function onSignUpStore(SignUpRequest $request){
        try {
            $redirectTo = request('returnUrl') ?? route('website-home');
            $data = [
                "education_level_id"    => EducationLevel::first()->id,
                "name_latin"            => $request->name,
                "name_khmer"            => $request->name,
                "dob"                   => null,
                "id_card"               => null,
                "sid"                   => null,
                "phone"                 => $request->phone,
                "phone_with_telegram"   => $request->phone,
                "gender"                => 'M',
                "email"                 => $request->email,
                "password"              => bcrypt($request->password),
                "address_khmer"         => null,
                "address_latin"         => null,
                "noted"                 => null,
                "status"                => 1,
                "approval_status"       => "pending",
                "current_working_place" => null,
                "working_experience"    => null,
            ];
            Student::create($data);

            auth('web')->attempt([
                'email'     => $request->email,
                'password'  => $request->password,
            ]);

            return response()->json([
                'status'   => true,
                'error'    => false,
                'message'  => 'Register success',
                'redirectTo' => $redirectTo,
            ]);
            
        } catch (\Exception $e) {
            auth('web')->logout();
            return response()->json([
                'status'   => false,
                'error'    => true,
                'message'  => $e->getMessage(),
            ]);
        }
    }

    public function onForgotPassword()
    {
        return view($this->layout.'forgot-password');
    }

    public function onResetPassword(ResetPasswordRequest $request)
    {
        try {
            $redirectTo = request('returnUrl') ?? route('website-home');
            $user = Student::query()
                        ->whereEmail($request->email)
                        ->whereRole('student')
                        ->first();

            if (!$user) {
                return response()->json([
                    'status'   => false,
                    'error'    => true,
                    'message'  => 'Your account is not found',
                ]);
            }

            $user->password = bcrypt($request->password);
            $user->save();

            auth('web')->attempt([
                'email'     => $request->email,
                'password'  => $request->password,
            ]);

            return response()->json([
                'status'   => true,
                'error'    => false,
                'message'  => 'Reset password success',
                'redirectTo' => $redirectTo,
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status'   => false,
                'error'    => true,
                'message'  => $e->getMessage(),
            ]);
        }
    }

    public function onSocial(Request $request)
    {
        try {
            $redirectTo = request('returnUrl') ?? route('website-home');
            $user = Student::query()
                        ->whereEmail($request->email)
                        ->whereRole('student')
                        ->first();

            $items = [
                "education_level_id"    => EducationLevel::first()->id,
                "name_latin"            => $request->name,
                "name_khmer"            => $request->name,
                "dob"                   => null,
                "id_card"               => null,
                "sid"                   => null,
                "phone"                 => $request->phone,
                "phone_with_telegram"   => $request->phone,
                "gender"                => 'M',
                "email"                 => $request->email,
                "password"              => bcrypt("12345678"),
                "address_khmer"         => null,
                "address_latin"         => null,
                "noted"                 => null,
                "status"                => 1,
                "approval_status"       => "pending",
                "current_working_place" => null,
                "working_experience"    => null,
            ];

            if(empty($user)){
                $createUser = User::create($items);
                auth('web')->login($createUser);
            }else{
                auth('web')->login($user);
            }

            return response()->json([
                'status'        => true,
                'error'         => false,
                'message'       => 'Login success',
                'redirectTo'    => $redirectTo,
            ]);

        }catch (\Exception $e) {
            return response()->json([
                'status'        => false,
                'error'         => true,
                'message'       => $e->getMessage(),
            ]);
        }
    }

    public function onSignOut()
    {
        try {
            auth()->guard('web')->logout();
            session()->flash('success', 'You are signed out!');
            return redirect()->route('home');

        } catch (\Exception $e) {
            session()->flash('error', 'Something went wrong!');
            return back();
        }
    }
}